"use client";
import React, { useEffect, useState } from "react";
import {
  useGetAssignmentByIdQuery,
  useSubmitAssignmentMutation,
} from "@/services/Api";
import { useSearchParams } from "next/navigation";
import { LuLoader2 } from "react-icons/lu";
import ResultModal from "./ResultModal";
import moment from "moment/moment";
import Notifications from "../../notifications/Notifications";
import SearchBox from "../../searchBox/SearchBox";
import { useRouter } from 'next/navigation';


const FillAssignment = () => {
  const router = useRouter();
  const searchParams = useSearchParams();
  const videoId = searchParams.get("video_id");
  const assignmentId = searchParams.get("assignment_id");
  const [isReslutModalVisible, setResultModalVisible] = useState(false);
  const [shuffledQuestions, setShuffledQuestions] = useState([]);
  const [selectedOptions, setSelectedOptions] = useState({});
  const countdownDuration = 10;
  const [timeLeft, setTimeLeft] = useState(countdownDuration);
  const [result, setResult] = useState(null);

  const {
    data: getAssignmentById,
    refetch: getAssignmentByIdRefetch,
    isLoading: getAssignmentByIdIsLoading,
  } = useGetAssignmentByIdQuery({ id: assignmentId, vid: videoId });

  const [submitAssignment, { isLoading: submitAssignmentLoading }] =
    useSubmitAssignmentMutation();

  const handleOptionChange = (questionIndex, selectedOption) => {
    setSelectedOptions((prev) => ({
      ...prev,
      [questionIndex]: [selectedOption],
    }));
  };

  const handleCheckboxChange = (questionIndex, selectedOption) => {
    setSelectedOptions((prev) => {
      const currentOptions = prev[questionIndex] || [];

      if (currentOptions.includes(selectedOption)) {
        return {
          ...prev,
          [questionIndex]: currentOptions.filter(
            (option) => option !== selectedOption
          ),
        };
      } else {
        return {
          ...prev,
          [questionIndex]: [...currentOptions, selectedOption],
        };
      }
    });
  };

  const preparePayload = () => {
    const payload = {
      answers: getAssignmentById?.assignment?.questions?.reduce(
        (acc, question, questionIndex) => {
          const selectedOptionIds = selectedOptions[questionIndex];

          if (selectedOptionIds && selectedOptionIds.length > 0) {
            acc.push({
              question_id: question.id,
              option_id: selectedOptionIds,
            });
          }

          return acc;
        },
        []
      ),
    };

    return payload;
  };

  const submitAssignmentApi = () => {
    const data = preparePayload();
    submitAssignment({ data: data.answers, videoId: videoId, assignmentId: assignmentId })
      .unwrap()
      .then((payload) => {
        if (payload?.final_result) {
          handleResultResultModal();
          setResult(true);
        } else {
          handleResultResultModal();
          setResult(false);
        }
      })
      .catch(() => {});
  };

  useEffect(() => {
    const savedStartTime = localStorage.getItem("countdownStartTime");
    const currentTime = Math.floor(Date.now() / 1000);

    if (!savedStartTime) {
      localStorage.setItem("countdownStartTime", currentTime);
    } else {
      const elapsedTime = currentTime - parseInt(savedStartTime, 10);
      const newTimeLeft = countdownDuration - elapsedTime;

      if (newTimeLeft <= 0) {
        setTimeLeft(0);
        localStorage.removeItem("countdownStartTime");
      } else {
        setTimeLeft(newTimeLeft);
      }
    }

    const interval = setInterval(() => {
      const currentTime = Math.floor(Date.now() / 1000);
      const savedStartTime = localStorage.getItem("countdownStartTime");
      const elapsedTime = currentTime - parseInt(savedStartTime, 10);
      const newTimeLeft = countdownDuration - elapsedTime;

      if (newTimeLeft <= 0) {
        setTimeLeft(0);
        localStorage.removeItem("countdownStartTime");
        clearInterval(interval);
      } else {
        setTimeLeft(newTimeLeft);
      }
    }, 1000);

    return () => clearInterval(interval);
  }, [countdownDuration]);

  const time = moment.utc(timeLeft * 1000).format("mm:ss");

  useEffect(() => {
    const handleBeforeUnload = (event) => {
      event.preventDefault();
      event.returnValue = "";
    };

    window.addEventListener("beforeunload", handleBeforeUnload);

    return () => {
      window.removeEventListener("beforeunload", handleBeforeUnload);
    };
  }, []);

  useEffect(() => {
    getAssignmentByIdRefetch();
  }, []);

  const handleResultResultModal = () => {
    setResultModalVisible(!isReslutModalVisible);
  };

  const shuffleArray = (array) => {
    const shuffledArray = [...array];
    for (let i = shuffledArray.length - 1; i > 0; i--) {
      const j = Math.floor(Math.random() * (i + 1));
      [shuffledArray[i], shuffledArray[j]] = [
        shuffledArray[j],
        shuffledArray[i],
      ];
    }
    return shuffledArray;
  };

  useEffect(() => {
    if (getAssignmentById?.assignment?.questions) {
      const shuffled = shuffleArray(getAssignmentById.assignment.questions).map(
        (question) => ({
          ...question,
          questionoptions: shuffleArray(question.questionoptions),
        })
      );
      setShuffledQuestions(shuffled);
    }
  }, [getAssignmentById]);

  return (
    <div className="md:px-[28px] px-3 pb-8 min-h-screen">
      <h1 className="py-2 text-[34px] font-bold pt-5 flex item-center">
        <span
          onClick={() => router.back()}
          className="text-[24px] pt-2 pe-2  cursor-pointer"
        >←  </span>
        Assignment
      </h1>
      <>
        <div className="flex items-center justify-between gap-8">
          <div className="md:w-2/3">
            <SearchBox />
          </div>
          <div className="flex items-center gap-2 text-darkgrey">
            <Notifications />
          </div>
        </div>
        {getAssignmentByIdIsLoading ? (
          <div className="w-full flex justify-center items-center py-40">
            <LuLoader2 className="animate-spin text-black h-6 w-6" />
          </div>
        ) : (
          <div className="bg-[#FFFAFA] p-4 mt-6 lg:p-8 rounded-xl">
            <div className="flex flex-col md:flex-row gap-4 lg:gap-4 xl:gap-14">
              <div className="flex justify-between items-center md:gap-4">
                <label className="text-[16px] font-bold">
                  Assignment Title:
                </label>
                <p className="text-[12px] py-2 px-3 w-[190px] rounded-lg font-bold border border-gray-400">
                  {getAssignmentById?.assignment?.title}
                </p>
              </div>
              <div className="flex items-center justify-between md:gap-4">
                <label className="text-[16px] font-bold">Duration:</label>
                <p className="text-[12px] py-2 px-3 w-[190px] rounded-lg font-bold border border-gray-400">
                  {getAssignmentById?.assignment?.duration} min
                </p>
              </div>
              <div className="flex items-center justify-between md:gap-4">
                <label className="text-[16px] font-bold">Time Left:</label>
                <p className="text-[12px] py-2 px-3 w-[190px] rounded-lg font-bold border border-gray-400">
                  {time} min
                </p>
              </div>
            </div>
            {shuffledQuestions?.map((question, questionIndex) => (
              <div
                key={questionIndex}
                className="mt-4 p-3 rounded-lg border border-gray-400"
              >
                <div className="flex gap-2 justify-between items-center">
                  <div className="flex sm:gap-4 w-full">
                    <div className="flex gap-1 items-center">
                      <label className="text-[16px] min-w-28 font-bold">
                        Question {questionIndex + 1}:
                      </label>
                    </div>
                    <p className="rounded-lg border py-3 px-3 w-2/3 border-gray-400">
                      {question?.title}
                    </p>
                  </div>
                </div>

                <div className="flex sm:flex-col justify-around xl:pl-28 lg:pl-20 md:pl-10 mt-6">
                  <div className="sm:flex gap-6 md:items-center sm:mt-0">
                    <div>
                      <span>Current Choice:</span>
                    </div>
                    <div className="flex sm:flex-row flex-col gap-8">
                      {question?.questionoptions?.map((option, optionIndex) => (
                        <div
                          className="flex items-center gap-1"
                          key={optionIndex}
                        >
                          {question?.question_answers_count > 1 ? (
                            <input
                              type="checkbox"
                              id={`option-${questionIndex}-${optionIndex}`}
                              name={`question-${questionIndex}`}
                              value={option.id}
                              checked={
                                selectedOptions[questionIndex]?.includes(
                                  option.id
                                ) || false
                              }
                              onChange={() =>
                                handleCheckboxChange(questionIndex, option.id)
                              }
                            />
                          ) : (
                            <input
                              type="radio"
                              id={`option-${questionIndex}-${optionIndex}`}
                              name={`question-${questionIndex}`}
                              value={option.id}
                              checked={
                                selectedOptions[questionIndex]?.[0] ===
                                option.id
                              }
                              onChange={() =>
                                handleOptionChange(questionIndex, option.id)
                              }
                            />
                          )}
                          <label
                            htmlFor={`option-${questionIndex}-${optionIndex}`}
                          >
                            {option?.value}
                          </label>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            ))}

            <button
              className="mt-4 py-2 px-4 rounded-lg bg-golden text-white"
              disabled={submitAssignmentLoading}
              onClick={submitAssignmentApi}
            >
              {submitAssignmentLoading ? "Loading..." : "Submit Assignment"}
            </button>
          </div>
        )}
      </>

      <ResultModal
        result={result}
        isVisible={isReslutModalVisible}
        onClose={() => setResultModalVisible(false)}
      />
    </div>
  );
};

export default FillAssignment;
