"use client";
import React, { useState } from "react";
import { Formik, Form, Field, ErrorMessage } from "formik";
import * as Yup from "yup";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
  faPhone,
  faEnvelope,
  faLocationDot,
} from "@fortawesome/free-solid-svg-icons";
import {
  faFacebook,
  faInstagram,
  faLinkedin,
} from "@fortawesome/free-brands-svg-icons";
import Link from "next/link";
import Image from "next/image";
import { BASE_URL } from "@/services/ApiEndpoints";

// Validation schema using Yup
const validationSchema = Yup.object({
  first_name: Yup.string()
    .min(2, "First name must be at least 2 characters")
    .required("First name is required"),
  last_name: Yup.string()
    .min(2, "Last name must be at least 2 characters")
    .required("Last name is required"),
  email: Yup.string()
    .email("Invalid email address")
    .required("Email is required"),
  phone: Yup.string()
    .matches(/^\+?[\d\s-]{10,}$/, "Invalid phone number")
    .required("Phone number is required"),
  message: Yup.string()
    .required("Message is required"),
});

const ContactUs = () => {
  const [formStatus, setFormStatus] = useState(null); // Track submission status

  // Initial form values
  const initialValues = {
    first_name: "",
    last_name: "",
    email: "",
    phone: "",
    message: "",
  };

  // Handle form submission
  const handleSubmit = async (values, { setSubmitting, resetForm }) => {
    try {
      const response = await fetch(`${BASE_URL}contact`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(values),
      });

      if (response.ok) {
        setFormStatus({ type: "success", message: "Message sent successfully!" });
        resetForm(); // Clear form fields
      } else {
        const errorData = await response.json();
        setFormStatus({
          type: "error",
          message: errorData.message || "Failed to send message.",
        });
      }
    } catch (error) {
      setFormStatus({
        type: "error",
        message: "An error occurred. Please try again later.",
      });
    } finally {
      setSubmitting(false);
    }
  };

  return (
    <div>
      {/* Optimized Image with Next.js Image Component */}
      <div className="relative">
        <Image
          src="/assets/home/homeCard.png"
          alt="Home Card"
          width={1920}
          height={500}
          className="w-full max-h-[500px] object-cover"
          priority
        />
        <div className="absolute top-0 left-0 h-full w-full flex justify-center items-center bg-black opacity-60">
          <p className="text-white font-medium text-3xl">Contact Us</p>
        </div>
      </div>

      <div className="max-w-7xl py-10 px-6 mx-auto">
        <div className="flex gap-8">
          {/* Left Part */}
          <div className="left-part flex flex-col justify-between bg-[#EFF4FF] max-w-[480px] rounded-lg min-h-full p-6">
            <div>
              <h1 className="text-[#1E1E1E] text-3xl font-semibold">
                Contact Us
              </h1>
              <p className="text-[#1E1E1E] text-base font-normal my-6">
                Lorem Ipsum is simply dummy text of the printing and typesetting
                industry. Lorem Ipsum has been the industry's standard dummy text
                ever since the 1500s,
              </p>

              <div className="flex flex-col gap-8 mt-20">
                <div className="flex items-center gap-4 text-[#1E1E1E]">
                  <FontAwesomeIcon
                    icon={faPhone}
                    className="text-golden h-5 w-5"
                  />
                  <span>864.672.4675</span>
                </div>
                <div className="flex items-center gap-4 text-[#1E1E1E]">
                  <FontAwesomeIcon
                    icon={faEnvelope}
                    className="text-golden h-5 w-5"
                  />
                  <span>info@vanrockre.com</span>
                </div>
                <div className="flex gap-4 text-[#1E1E1E]">
                  <FontAwesomeIcon
                    icon={faLocationDot}
                    className="text-golden h-6 w-6"
                  />
                  <span className="w-44">
                    333 Wade Hampton Blvd Greenville, SC 29609
                  </span>
                </div>
              </div>
            </div>
            <div className="flex gap-5">
              <Link href="https://facebook.com" target="_blank" rel="noopener noreferrer">
                <FontAwesomeIcon
                  icon={faFacebook}
                  className="h-8 w-8 text-[#1E1E1E]"
                />
              </Link>
              <Link href="https://instagram.com" target="_blank" rel="noopener noreferrer">
                <FontAwesomeIcon
                  icon={faInstagram}
                  className="h-8 w-8 text-[#1E1E1E]"
                />
              </Link>
              <Link href="https://linkedin.com" target="_blank" rel="noopener noreferrer">
                <FontAwesomeIcon
                  icon={faLinkedin}
                  className="h-8 w-8 text-[#1E1E1E]"
                />
              </Link>
            </div>
          </div>

          {/* Right Part: Formik Form */}
          <div className="right-part w-full">
            <Formik
              initialValues={initialValues}
              validationSchema={validationSchema}
              onSubmit={handleSubmit}
            >
              {({ isSubmitting }) => (
                <Form noValidate>
                  {/* First Name */}
                  <div className="flex flex-col gap-4 mb-6">
                    <label className="text-[#1E1E1E] text-lg font-normal">
                      First Name
                    </label>
                    <Field
                      type="text"
                      name="first_name"
                      placeholder="Enter First Name"
                      className="w-full p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                    />
                    <ErrorMessage
                      name="first_name"
                      component="p"
                      className="text-red-500 text-sm"
                    />
                  </div>

                  {/* Last Name */}
                  <div className="flex flex-col gap-4 mb-6">
                    <label className="text-[#1E1E1E] text-lg font-normal">
                      Last Name
                    </label>
                    <Field
                      type="text"
                      name="last_name"
                      placeholder="Enter Last Name"
                      className="w-full p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                    />
                    <ErrorMessage
                      name="last_name"
                      component="p"
                      className="text-red-500 text-sm"
                    />
                  </div>

                  {/* Email */}
                  <div className="flex flex-col gap-4 mb-6">
                    <label className="text-[#1E1E1E] text-lg font-normal">
                      Email
                    </label>
                    <Field
                      type="email"
                      name="email"
                      placeholder="Enter Email"
                      className="w-full p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                    />
                    <ErrorMessage
                      name="email"
                      component="p"
                      className="text-red-500 text-sm"
                    />
                  </div>

                  {/* Phone Number */}
                  <div className="flex flex-col gap-4 mb-6">
                    <label className="text-[#1E1E1E] text-lg font-normal">
                      Phone Number
                    </label>
                    <Field
                      type="text"
                      name="phone"
                      placeholder="Enter Phone Number"
                      className="w-full p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                    />
                    <ErrorMessage
                      name="phone"
                      component="p"
                      className="text-red-500 text-sm"
                    />
                  </div>

                  {/* Message */}
                  <div className="flex flex-col gap-4 mb-6">
                    <label className="text-[#1E1E1E] text-lg font-normal">
                      Message
                    </label>
                    <Field
                      as="textarea"
                      name="message"
                      placeholder="Enter Message"
                      className="w-full p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                      rows="8"
                    />
                    <ErrorMessage
                      name="message"
                      component="p"
                      className="text-red-500 text-sm"
                    />
                  </div>

                  {/* Form Status */}
                  {formStatus && (
                    <p
                      className={`mb-4 text-sm ${
                        formStatus.type === "success"
                          ? "text-green-500"
                          : "text-red-500"
                      }`}
                    >
                      {formStatus.message}
                    </p>
                  )}

                  {/* Submit Button */}
                  <button
                    type="submit"
                    className="bg-golden text-white py-2 px-3 rounded-md text-lg font-medium disabled:opacity-50"
                    disabled={isSubmitting}
                  >
                    {isSubmitting ? "Submitting..." : "Submit"}
                  </button>
                </Form>
              )}
            </Formik>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ContactUs;