"use client";
import React from "react";
import { Formik, Form, Field, ErrorMessage } from "formik";
import * as Yup from "yup";
import { BASE_URL } from "@/services/ApiEndpoints";
import toast from "react-hot-toast";

// Validation schema using Yup
const validationSchema = Yup.object({
  name: Yup.string().required("Name is required"),
  email: Yup.string().email("Invalid email address").required("Email is required"),
  q3: Yup.string().required("Please select an option"),
  q4: Yup.string().required("Please select an option"),
  q5: Yup.string().when("q4", {
    is: "Yes",
    then: (schema) => schema.required("Please describe your challenge"),
    otherwise: (schema) => schema.optional(),
  }),
  q6: Yup.string().required("Please provide an answer"),
  q7: Yup.string().required("Please select an option"),
  q8: Yup.string().required("Please select an option"),
  q9: Yup.string().when("q8", {
    is: (value) => ["Maybe", "No"].includes(value),
    then: (schema) => schema.required("Please explain your choice"),
    otherwise: (schema) => schema.optional(),
  }),
  q10: Yup.array().min(1, "Please select at least one option"),
  q11: Yup.string().required("Please provide an answer"),
  q12: Yup.string().required("Please provide an answer"),
});

const QuickSurvey = () => {
  const initialValues = {
    name: "",
    email: "",
    q3: "",
    q4: "",
    q5: "",
    q6: "",
    q7: "",
    q8: "",
    q9: "",
    q10: [],
    q11: "",
    q12: "",
  };

  
    // Handle form submission
    const handleSubmit = async (values, { setSubmitting, resetForm }) => {
      try {
        const response = await fetch(`${BASE_URL}survey`, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(values),
        });
  
        if (response.ok) {
          let data = await response.json();
          resetForm();
          toast.success(data.message, {
            position: "top-center",
          });
        } 
      } catch (error) {
        toast.error("Survey not sent!", {
          position: "top-center",
        });
      }
    };
    

  return (
    <div>
      <div className="max-w-7xl py-10 px-6 mx-auto">
        <h1 className="text-[#1E1E1E] text-3xl font-semibold text-center mb-16">
          Quick Survey
        </h1>
        <div className="flex gap-8">
          <div className="w-full">
            <Formik
              initialValues={initialValues}
              validationSchema={validationSchema}
              onSubmit={handleSubmit}
            >
              {({ isSubmitting, values }) => (
                <Form>
                  <div className="flex flex-col gap-4 mb-6">
                    {/* Question 1: Name */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        1) Name
                      </label>
                      <Field
                        type="text"
                        name="name"
                        placeholder="Please enter your full name."
                        className="w-full mt-3 p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                      />
                      <ErrorMessage
                        name="name"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 2: Email */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        2) Email
                      </label>
                      <Field
                        type="email"
                        name="email"
                        placeholder="Please enter your email address."
                        className="w-full mt-3 p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                      />
                      <ErrorMessage
                        name="email"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 3: Considering a career in real estate */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        3) Are you currently considering a career in real estate?
                      </label>
                      <div className="flex flex-col pt-3 ps-2">
                        <label className="custom-radio mb-2 flex gap-2">
                          <Field type="radio" name="q3" value="Yes" />
                          Yes
                        </label>
                        <label className="custom-radio mb-2 flex gap-2">
                          <Field type="radio" name="q3" value="No" />
                          No
                        </label>
                      </div>
                      <ErrorMessage
                        name="q3"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 4: Researched pre-licensing courses */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        4) Have you previously taken or researched real estate
                        pre-licensing courses?
                      </label>
                      <div className="flex flex-col pt-3 ps-2">
                        <label className="custom-radio mb-2 flex gap-2">
                          <Field type="radio" name="q4" value="Yes" />
                          Yes
                        </label>
                        <label className="custom-radio mb-2 flex gap-2">
                          <Field type="radio" name="q4" value="No" />
                          No
                        </label>
                      </div>
                      <ErrorMessage
                        name="q4"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 5: Challenges with the process */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        5) If yes, what was your biggest challenge or frustration
                        with the process?
                      </label>
                      <Field
                        as="textarea"
                        name="q5"
                        className="w-full mt-3 p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                        rows="3"
                      />
                      <ErrorMessage
                        name="q5"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 6: Importance of structured education */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        6) How important is having a structured, success-driven
                        education experience in your real estate journey?
                      </label>
                      <Field
                        as="textarea"
                        name="q6"
                        className="w-full mt-3 p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                        rows="3"
                      />
                      <ErrorMessage
                        name="q6"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 7: Expected cost */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        7) How much do you expect to spend on a real estate
                        pre-licensing education?
                      </label>
                      <div className="flex flex-col pt-3 ps-2">
                        {[
                          "$0-$500",
                          "$500-$1,500",
                          "$1,500-$3,000",
                          "$3,000-$5,000",
                          "$5,000+",
                        ].map((option) => (
                          <label
                            key={option}
                            className="custom-radio mb-2 flex gap-2"
                          >
                            <Field type="radio" name="q7" value={option} />
                            {option}
                          </label>
                        ))}
                      </div>
                      <ErrorMessage
                        name="q7"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 8: Willingness to pay $5,000 */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        8) Would you be willing to pay $5,000 for a pre-licensing
                        program that not only helps you pass the exam but also
                        sets you up for long-term success in real estate?
                      </label>
                      <div className="flex flex-col pt-3 ps-2">
                        <label className="custom-radio mb-2 flex gap-2">
                          <Field type="radio" name="q8" value="Yes" />
                          Yes
                        </label>
                        <label className="custom-radio mb-2 flex gap-2">
                          <Field type="radio" name="q8" value="Maybe" />
                          Maybe
                        </label>
                        <label className="custom-radio mb-2 flex gap-2">
                          <Field type="radio" name="q8" value="No" />
                          No
                        </label>
                      </div>
                      <ErrorMessage
                        name="q8"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 9: Justification for high-priced course */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        9) If no or maybe, what would make a high-priced course
                        worth the investment for you?
                      </label>
                      <Field
                        as="textarea"
                        name="q9"
                        className="w-full mt-3 p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                        rows="3"
                      />
                      <ErrorMessage
                        name="q9"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 10: Additional benefits */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        10) What additional benefits would justify paying a 10x
                        premium for a real estate education program? (Select all
                        that apply)
                      </label>
                      <div className="flex flex-col pt-3 ps-2">
                        {[
                          "Mentorship from top real estate professionals",
                          "Hands-on training & real-world scenarios",
                          "Marketing & lead generation training",
                          "Post-license coaching & brokerage connections",
                          "Other",
                        ].map((option) => (
                          <label
                            key={option}
                            className="custom-radio mb-2 flex gap-2"
                          >
                            <Field type="checkbox" name="q10" value={option} />
                            {option}
                          </label>
                        ))}
                      </div>
                      <ErrorMessage
                        name="q10"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 11: Biggest concern */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        11) What is your biggest concern about investing in a
                        high-priced real estate pre-licensing program?
                      </label>
                      <Field
                        as="textarea"
                        name="q11"
                        className="w-full mt-3 p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                        rows="3"
                      />
                      <ErrorMessage
                        name="q11"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>

                    {/* Question 12: Confidence in premium course */}
                    <div className="mb-3">
                      <label className="text-[#1E1E1E] text-lg font-normal">
                        12) What would make you feel confident that a
                        premium-priced course is worth the investment?
                      </label>
                      <Field
                        as="textarea"
                        name="q12"
                        className="w-full mt-3 p-3 rounded-lg border border-[#E3E3E3] focus:border-golden outline-none"
                        rows="3"
                      />
                      <ErrorMessage
                        name="q12"
                        component="div"
                        className="text-red-500 text-sm mt-1"
                      />
                    </div>
                  </div>
                  <button
                    type="submit"
                    disabled={isSubmitting}
                    className="bg-golden text-white py-2 px-3 rounded-md text-lg font-medium disabled:opacity-50"
                  >
                    Submit
                  </button>
                </Form>
              )}
            </Formik>
          </div>
        </div>
      </div>
    </div>
  );
};

export default QuickSurvey;