// File generated from our OpenAPI spec by Stainless. See CONTRIBUTING.md for details.
import { APIResource } from '@mux/mux-node/resource';
import crypto from 'crypto';
import { getRequiredHeader } from '@mux/mux-node/core';
export class Webhooks extends APIResource {
    /**
     * Validates that the given payload was sent by Mux and parses the payload
     */
    unwrap(body, headers, secret = this._client.webhookSecret) {
        this.verifySignature(body, headers, secret);
        const traverse = (value) => {
            if (Array.isArray(value)) {
                return value.map(traverse);
            }
            if (value !== null && typeof value === 'object') {
                // Mux's webhook event types differ from the normal API calls as instead of an ISO 8601 string
                // we return { nanos: 123456, second: 1234 }. We transform those objects into ISO 8601 strings
                // to conform to the normal api call types.
                if ('nanos' in value && 'seconds' in value) {
                    const timeObject = value;
                    const date = new Date((timeObject.nanos + timeObject.seconds * 1e6) / 1e3);
                    return date.toISOString();
                }
                return Object.fromEntries(Object.entries(value).map(([key, subValue]) => {
                    return [key, traverse(subValue)];
                }));
            }
            return value;
        };
        return traverse(JSON.parse(body));
    }
    parseHeader(header, scheme) {
        if (typeof header !== 'string') {
            return null;
        }
        return header.split(',').reduce((accum, item) => {
            const kv = item.split('=');
            if (kv[0] === 't') {
                /* eslint-disable no-param-reassign, prefer-destructuring */
                accum.timestamp = parseInt(kv[1], 10);
            }
            if (kv[0] === scheme && typeof kv[1] === 'string') {
                accum.signatures.push(kv[1]);
            }
            return accum;
        }, {
            timestamp: -1,
            signatures: [],
        });
    }
    /** Make an assertion, if not `true`, then throw. */
    assert(expr, msg = '') {
        if (!expr) {
            throw new Error(msg);
        }
    }
    computeSignature(payload, secret) {
        return crypto.createHmac('sha256', secret).update(payload, 'utf8').digest('hex');
    }
    /** Compare to array buffers or data views in a way that timing based attacks
     * cannot gain information about the platform. */
    timingSafeEqual(a, b) {
        if (a.byteLength !== b.byteLength) {
            return false;
        }
        if (!(a instanceof DataView)) {
            a = new DataView(ArrayBuffer.isView(a) ? a.buffer : a);
        }
        if (!(b instanceof DataView)) {
            b = new DataView(ArrayBuffer.isView(b) ? b.buffer : b);
        }
        this.assert(a instanceof DataView);
        this.assert(b instanceof DataView);
        const length = a.byteLength;
        let out = 0;
        let i = -1;
        while (++i < length) {
            out |= a.getUint8(i) ^ b.getUint8(i);
        }
        return out === 0;
    }
    /**
     * Validates whether or not the webhook payload was sent by Mux.
     *
     * If it was not sent by Mux then an error will be raised.
     */
    verifySignature(body, headers, secret = this._client.webhookSecret) {
        if (!secret) {
            throw new Error("The webhook secret must either be set using the env var, MUX_WEBHOOK_SECRET, on the client class, Mux({ webhookSecret: '123' }), or passed to this function");
        }
        const header = getRequiredHeader(headers, 'mux-signature');
        if (!header) {
            throw new Error('Could not find a mux-signature header');
        }
        if (typeof body !== 'string') {
            throw new Error('Webhook body must be passed as the raw JSON string sent from the server (do not parse it first).');
        }
        const details = this.parseHeader(header, 'v1');
        if (!details || details.timestamp === -1) {
            throw new Error('Unable to extract timestamp and signatures from header');
        }
        if (!details.signatures.length) {
            throw new Error('No v1 signatures found');
        }
        const expectedSignature = this.computeSignature(`${details.timestamp}.${body}`, secret);
        const encoder = new TextEncoder();
        const signatureFound = !!details.signatures.filter((sig) => this.timingSafeEqual(encoder.encode(sig), encoder.encode(expectedSignature))).length;
        if (!signatureFound) {
            throw new Error('No signatures found matching the expected signature for payload.');
        }
        const timestampAge = Math.floor(Date.now() / 1000) - details.timestamp;
        const tolerance = 300; // 5 minutes
        if (timestampAge > tolerance) {
            throw new Error('Webhook timestamp is too old');
        }
    }
}
(function (Webhooks) {
})(Webhooks || (Webhooks = {}));
//# sourceMappingURL=webhooks.mjs.map