export default createMediaStore;
export type RequestMap = import('./request-map.js').RequestMap;
export type StateMediator = import('./state-mediator.js').StateMediator;
/**
 * <D>
 */
export type EventOrAction<D> = import('./state-mediator.js').EventOrAction<D>;
export type MediaStateOwner = import('./state-mediator.js').MediaStateOwner;
export type FullScreenElementStateOwner = import('./state-mediator.js').FullScreenElementStateOwner;
export type RootNodeStateOwner = import('./state-mediator.js').RootNodeStateOwner;
export type StateOption = import('./state-mediator.js').StateOption;
export type StateOwners = import('./state-mediator.js').StateOwners;
/**
 * MediaState is a full representation of all media-related state modeled by the MediaStore and its StateMediator.
 * Instead of checking the StateOwners' state directly or on the fly, MediaStore keeps a "snapshot" of the latest
 * state, which will be provided to any MediaStore subscribers whenever the state changes, and is arbitrarily retrievable
 * from the MediaStore using `getState()`.
 */
export type MediaState = {
    readonly mediaPaused: boolean;
    readonly mediaHasPlayed: boolean;
    readonly mediaEnded: boolean;
    readonly mediaPlaybackRate: number;
    readonly mediaMuted: boolean;
    readonly mediaVolume: number;
    readonly mediaVolumeLevel: "high" | "medium" | "low" | "off";
    readonly mediaCurrentTime: number;
    readonly mediaDuration: number;
    readonly mediaLoading: boolean;
    readonly mediaSeekable: [number, number];
    readonly mediaBuffered: [number, number][];
    readonly mediaStreamType: import("./state-mediator.js").StreamTypeValue;
    readonly mediaTargetLiveWindow: number;
    readonly mediaTimeIsLive: boolean;
    readonly mediaSubtitlesList: Pick<TextTrack, "kind" | "language" | "label">[];
    readonly mediaSubtitlesShowing: Pick<TextTrack, "kind" | "language" | "label">[];
    readonly mediaChaptersCues: Pick<VTTCue, "text" | "startTime" | "endTime">[];
    readonly mediaIsPip: boolean;
    readonly mediaRenditionList: import("./state-mediator.js").Rendition[];
    readonly mediaRenditionSelected: string;
    readonly mediaAudioTrackList: {
        id?: string;
    }[];
    readonly mediaAudioTrackEnabled: string;
    readonly mediaIsFullscreen: boolean;
    readonly mediaIsCasting: boolean;
    readonly mediaIsAirplaying: boolean;
    readonly mediaFullscreenUnavailable: import("./state-mediator.js").AvailabilityTypeValue;
    readonly mediaPipUnavailable: import("./state-mediator.js").AvailabilityTypeValue;
    readonly mediaVolumeUnavailable: import("./state-mediator.js").AvailabilityTypeValue;
    readonly mediaCastUnavailable: import("./state-mediator.js").AvailabilityTypeValue;
    readonly mediaAirplayUnavailable: import("./state-mediator.js").AvailabilityTypeValue;
    readonly mediaRenditionUnavailable: import("./state-mediator.js").AvailabilityTypeValue;
    readonly mediaAudioTrackUnavailable: import("./state-mediator.js").AvailabilityTypeValue;
    readonly mediaPreviewTime: number;
    readonly mediaPreviewImage: string;
    readonly mediaPreviewCoords: [string, string, string, string];
};
/**
 * ;
 */
export type MediaStore = {
    /**
     * A method that expects an "Action" or "Event". Primarily used to make state change requests.
     */
    dispatch: (eventOrAction: EventOrAction<any>) => void;
    /**
     * A method to get the current state of the MediaStore
     */
    getState: () => Partial<MediaState>;
    /**
     * A method to "subscribe" to the MediaStore. A subscriber is just a callback function that is invoked with the current state whenever the MediaStore's state changes. The method returns an "unsubscribe" function, which should be used to tell the MediaStore to remove the corresponding subscriber.
     */
    subscribe: (handler: (state: Partial<MediaState>) => void) => (() => void);
};
/**
 * @typedef {import('./request-map.js').RequestMap} RequestMap
 */
/**
 * @typedef {import('./state-mediator.js').StateMediator} StateMediator
 */
/**
 * @typedef {import('./state-mediator.js').EventOrAction<D>} EventOrAction<D>
 * @template D
 */
/**
 * @typedef {import('./state-mediator.js').MediaStateOwner} MediaStateOwner
 */
/**
 * @typedef {import('./state-mediator.js').FullScreenElementStateOwner} FullScreenElementStateOwner
 */
/**
 * @typedef {import('./state-mediator.js').RootNodeStateOwner} RootNodeStateOwner
 */
/**
 * @typedef {import('./state-mediator.js').StateOption} StateOption
 */
/**
 * @typedef {import('./state-mediator.js').StateOwners} StateOwners
 */
/**
 * MediaState is a full representation of all media-related state modeled by the MediaStore and its StateMediator.
 * Instead of checking the StateOwners' state directly or on the fly, MediaStore keeps a "snapshot" of the latest
 * state, which will be provided to any MediaStore subscribers whenever the state changes, and is arbitrarily retrievable
 * from the MediaStore using `getState()`.
 * @typedef {Readonly<{
 *   [K in keyof StateMediator]: ReturnType<StateMediator[K]['get']>
 * } & {
 *   mediaPreviewTime: number;
 *   mediaPreviewImage: string;
 *   mediaPreviewCoords: [string,string,string,string]
 * }>} MediaState
 */
/**
 * MediaStore is the primary abstraction for managing and monitoring media state and other state relevant to the media UI
 * (for example, fullscreen behavior or the availability of media-related functionality for a particular browser or runtime, such as volume control or Airplay). This includes:
 * - Keeping track of any state changes (examples: Is the media muted? Is the currently playing media live or on demand? What audio tracks are available for the current media?)
 * - Sharing the latest state with any MediaStore subscribers whenever it changes
 * - Receiving and responding to requests to change the media or related state (examples: I would like the media to be unmuted. I want to start casting now. I want to switch from English subtitles to Japanese.)
 * - Wiring up and managing the relationships between media state, media state change requests, and the stateful entities that “own” the majority of this state (examples: the current media element being used, the current root node, the current fullscreen element)
 * - Respecting and monitoring changes in certain optional behaviors that impact state or state change requests (examples: I want subtitles/closed captions to be on by default whenever media with them are loaded. I want to disable keeping track of the user’s preferred volume level.)
 *
 * @typedef {Object} MediaStore;
 * @property {(eventOrAction: EventOrAction<any>) => void} dispatch                       A method that expects an "Action" or "Event". Primarily used to make state change requests.
 * @property {() => Partial<MediaState>} getState                                         A method to get the current state of the MediaStore
 * @property {(handler: (state: Partial<MediaState>) => void) => (() => void)} subscribe  A method to "subscribe" to the MediaStore. A subscriber is just a callback function that is invoked with the current state whenever the MediaStore's state changes. The method returns an "unsubscribe" function, which should be used to tell the MediaStore to remove the corresponding subscriber.
 *
 * @example &lt;caption>Basic Usage.&lt;/caption>
 * const mediaStore = createStore({
 *   media: myVideoElement,
 *   fullscreenElement: myMediaUIContainerElement,
 *   // documentElement: advancedRootNodeCase // Will default to `document`
 *   options: {
 *     defaultSubtitles: true // enable subtitles/captions by default
 *   },
 * });
 *
 * // NOTE: In a more realistic example, `myToggleMutedButton` and `mySeekForwardButton` would likely keep track of/"own" its current state. See, e.g. the `<mute-button>` Media Chrome Web Component.
 * const unsubscribe = mediaStore.subscribe(state => {
 *   myToggleMutedButton.textContent = state.muted ? 'Unmute' : 'Mute';
 * });
 *
 * myToggleMutedButton.addEventListener('click', () => {
 *   const type = mediaStore.getState().muted ? 'mediaunmuterequest' : 'mediamuterequest'
 *   mediaStore.dispatch({ type });
 * });
 *
 * mySeekForwardButton.addEventListener('click', () => {
 *   mediaStore.dispatch({
 *     type: 'mediaseekrequest',
 *     // NOTE: For all of our state change requests that require additional information, we rely on the `detail` property so we can conform to `CustomEvent`, making interop easier.
 *     detail: mediaStore.getState().mediaCurrentTime + 15,
 *   });
 * });
 *
 * // If your code has cases where it swaps out the media element being used
 * mediaStore.dispatch({
 *   type: 'mediaelementchangerequest',
 *   detail: myAudioElement,
 * });
 *
 * // ... Eventual teardown, when relevant. This is especially relevant for potential garbage collection/memory management considerations.
 * unsubscribe();
 *
 */
/**
 * A factory for creating a `MediaStore` instance.
 * @param {{
 *   media?: MediaStateOwner;
 *   fullscreenElement?: FullScreenElementStateOwner;
 *   documentElement?: RootNodeStateOwner;
 *   stateMediator?: StateMediator;
 *   requestMap?: RequestMap;
 *   options?: StateOption;
 *   monitorStateOwnersOnlyWithSubscriptions?: boolean;
 * }} mediaStoreConfig
 * @returns {MediaStore}
 */
declare function createMediaStore({ media, fullscreenElement, documentElement, stateMediator, requestMap, options, monitorStateOwnersOnlyWithSubscriptions, }: {
    media?: MediaStateOwner;
    fullscreenElement?: FullScreenElementStateOwner;
    documentElement?: RootNodeStateOwner;
    stateMediator?: StateMediator;
    requestMap?: RequestMap;
    options?: StateOption;
    monitorStateOwnersOnlyWithSubscriptions?: boolean;
}): MediaStore;
