/** @typedef {import('../media-controller.js').MediaController} MediaController */
/**
 * Get the media controller element from the `mediacontroller` attribute or closest ancestor.
 * @param  {HTMLElement} host
 * @return {MediaController | undefined}
 */
export function getMediaController(host: HTMLElement): MediaController | undefined;
/**
 * Get the media controller element from the `mediacontroller` attribute.
 * @param  {HTMLElement} host
 * @return {MediaController | undefined}
 */
export function getAttributeMediaController(host: HTMLElement): MediaController | undefined;
/**
 * Get the active element, accounting for Shadow DOM subtrees.
 * @param {Document|ShadowRoot} root
 * @return {Element|null}
 */
export function getActiveElement(root?: Document | ShadowRoot): Element | null;
/**
 * Gets the document or shadow root of a node, not the node itself which can lead to bugs.
 * https://developer.mozilla.org/en-US/docs/Web/API/Node/getRootNode#return_value
 * @param {Node} node
 * @return {Document|ShadowRoot|null}
 */
export function getDocumentOrShadowRoot(node: Node): Document | ShadowRoot | null;
/**
 * Checks if the element is visible includes opacity: 0 and visibility: hidden.
 * @param  {HTMLElement} element
 * @return {Boolean}
 */
export function isElementVisible(element: HTMLElement, depth?: number): boolean;
/**
 * Get progress ratio of a point on a line segment.
 * @param  {number} x
 * @param  {number} y
 * @param  {{ x: number, y: number }} p1
 * @param  {{ x: number, y: number }} p2
 * @return {number}
 */
export function getPointProgressOnLine(x: number, y: number, p1: {
    x: number;
    y: number;
}, p2: {
    x: number;
    y: number;
}): number;
export function distance(p1: any, p2: any): number;
/**
 * Get or insert a CSSStyleRule with a selector in an element containing <style> tags.
 * @param  {Element|ShadowRoot} styleParent
 * @param  {string} selectorText
 * @return {CSSStyleRule | {
 *   style: {
 *     setProperty: () => void,
 *     removeProperty: () => void,
 *     width?: string,
 *     height?: string,
 *     display?: string,
 *     transform?: string,
 *   },
 *   selectorText: string,
 * }}
 */
export function getOrInsertCSSRule(styleParent: Element | ShadowRoot, selectorText: string): CSSStyleRule | {
    style: {
        setProperty: () => void;
        removeProperty: () => void;
        width?: string;
        height?: string;
        display?: string;
        transform?: string;
    };
    selectorText: string;
};
/**
 * Get a CSSStyleRule with a selector in an element containing <style> tags.
 * @param  {Element|ShadowRoot} styleParent
 * @param  {(selectorText) => boolean} predicate
 * @return {CSSStyleRule | undefined}
 */
export function getCSSRule(styleParent: Element | ShadowRoot, predicate: (selectorText: any) => boolean): CSSStyleRule | undefined;
/**
 * Insert a CSSStyleRule with a selector in an element containing <style> tags.
 * @param  {Element|ShadowRoot} styleParent
 * @param  {string} selectorText
 * @return {CSSStyleRule | undefined}
 */
export function insertCSSRule(styleParent: Element | ShadowRoot, selectorText: string): CSSStyleRule | undefined;
/**
 * Gets the number represented by the attribute
 * @param {any} el (Should be an HTMLElement, but need any for SSR cases)
 * @param {string} attrName
 * @param {number} [defaultValue = Number.NaN]
 * @returns {number | undefined} Will return undefined if no attribute set
 */
export function getNumericAttr(el: any, attrName: string, defaultValue?: number): number | undefined;
/**
 * @param {any} el (Should be an HTMLElement, but need any for SSR cases)
 * @param {string} attrName
 * @param {number} value
 */
export function setNumericAttr(el: any, attrName: string, value: number): void;
/**
 * @param {any} el (Should be an HTMLElement, but need any for SSR cases)
 * @param {string} attrName
 * @returns {boolean}
 */
export function getBooleanAttr(el: any, attrName: string): boolean;
/**
 * @param {any} el (Should be an HTMLElement, but need any for SSR cases)
 * @param {string} attrName
 * @param {boolean} value
 */
export function setBooleanAttr(el: any, attrName: string, value: boolean): void;
/**
 * @param {any} el (Should be an HTMLElement, but need any for SSR cases)
 * @param {string} attrName
 */
export function getStringAttr(el: any, attrName: string, defaultValue?: any): any;
/**
 * @param {*} el (Should be an HTMLElement, but need any for SSR cases)
 * @param {string} attrName
 * @param {string} value
 */
export function setStringAttr(el: any, attrName: string, value: string): void;
export function updateIconText(svg: any, value: any, selector?: string): void;
export function getAllSlotted(el: any, name: any): any;
export function getSlotted(el: any, name: any): any;
export function containsComposedNode(rootNode?: {
    contains?: Node['contains'];
}, childNode?: Node): any;
export function closestComposedNode(childNode: any, selector: any): any;
export type MediaController = import('../media-controller.js').MediaController;
