"use client";
import React, { useState, useRef } from "react";
import Logo from "../../../public/assets/agent-ambition.png";
import Image from "next/image";
import Link from "next/link";
import { useFormik } from "formik";
import * as Yup from "yup";
import { format } from "date-fns";
import { registorAPI } from "@/redux/authSlice";
import { useDispatch } from "react-redux";
import { useRouter, useSearchParams } from "next/navigation";
import toast from "react-hot-toast";
import DatePicker from "react-datepicker";
import "react-datepicker/dist/react-datepicker.css";
import { IMaskInput } from "react-imask";

const RegistretionForm = () => {
  const [date, setDate] = useState(new Date());
  const [frontImage, setFrontImage] = useState(null);
  const [frontImagePreview, setFrontImagePreview] = useState(null);
  const [frontFileName, setFrontFileName] = useState(null);
  const [backImage, setBackImage] = useState(null);
  const [backImagePreview, setBackImagePreview] = useState(null);
  const [backFileName, setBackFileName] = useState(null);
  const frontFileInputRef = useRef(null);
  const backFileInputRef = useRef(null);
  const dispatch = useDispatch();
  const router = useRouter();
  const searchParams = useSearchParams();
  const [errorMessage, setErrorMessage] = useState("");
  const [loading, setLoading] = useState(false);

  const planId = searchParams.get("plan_id") || "1";

  const handleFileChange = (file, type) => {
    if (file) {
      const allowedTypes = ["image/jpeg", "image/png", "application/pdf"];
      if (allowedTypes.includes(file.type)) {
        if (type === "front") {
          setFrontImage(file);
          setFrontFileName(file.name);
          if (file.type.startsWith("image/")) {
            setFrontImagePreview(URL.createObjectURL(file));
          } else {
            setFrontImagePreview(null);
          }
          formik.setFieldValue("id_card_front", file);
        } else {
          setBackImage(file);
          setBackFileName(file.name);
          if (file.type.startsWith("image/")) {
            setBackImagePreview(URL.createObjectURL(file));
          } else {
            setBackImagePreview(null);
          }
          formik.setFieldValue("id_card_back", file);
        }
      } else {
        toast.error("Please upload a valid JPG, PNG, or PDF file.", {
          position: "top-center",
        });
      }
    }
  };

  const handleDragOver = (event) => {
    event.preventDefault();
    event.stopPropagation();
  };

  const handleDrop = (event, type) => {
    event.preventDefault();
    event.stopPropagation();
    const file = event.dataTransfer.files[0];
    if (file) handleFileChange(file, type);
  };

  const removeFile = (type) => {
    if (type === "front") {
      setFrontImage(null);
      setFrontFileName(null);
      setFrontImagePreview(null);
      formik.setFieldValue("id_card_front", null);
    } else {
      setBackImage(null);
      setBackFileName(null);
      setBackImagePreview(null);
      formik.setFieldValue("id_card_back", null);
    }
  };

  const submitHandler = async (values) => {
    setErrorMessage("");
    setLoading(true);
    const formData = new FormData();
    formData.append("first_name", values.first_name || "");
    formData.append("last_name", values.last_name || "");
    formData.append("email", values.email || "");
    formData.append("contact_number", values.contact_number || "");
    formData.append("dob", values.dob || "");
    formData.append("timeslot_id", values.timeslot_id || "");
    formData.append("learning_style", values.learning_style || "");
    formData.append("address", values.address || "");
    formData.append("city", values.city || "");
    formData.append("state", values.state || "");
    formData.append("zip_code", values.zip_code || "");
    formData.append("plan_id", planId || "1");
    if (values.id_card_front) {
      formData.append("id_card_front", values.id_card_front);
    }
    if (values.id_card_back) {
      formData.append("id_card_back", values.id_card_back);
    }

    console.log("FormData Contents:");
    for (let [key, value] of formData.entries()) {
      console.log(`${key}:`, value instanceof File ? value.name : value);
    }

    try {
      const response = await dispatch(registorAPI(formData)).unwrap();
      if (response?.url) {
        setTimeout(() => {
          router.push(response.url);
        }, 3000);
      } else if (response?.status === "timeslot_full") {
        setErrorMessage(response.message);
        toast.error(response.message, { position: "top-center" });
      } else if (
        response?.status === "validation_failed" &&
        response?.errors?.email
      ) {
        setErrorMessage(response.errors.email[0]);
        toast.error(response.errors.email[0], { position: "top-center" });
      } else if (response?.status === "validation_failed") {
        const errors = Object.values(response.errors).flat().join(", ");
        setErrorMessage(errors);
        toast.error(errors, { position: "top-center" });
      } else {
        setErrorMessage("An unexpected error occurred. Please try again.");
        toast.error("An unexpected error occurred. Please try again.", {
          position: "top-center",
        });
      }
    } catch (error) {
      console.error("Registration error:", error);
      const errorMsg =
        error?.message || "Registration failed. Please try again later.";
      setErrorMessage(errorMsg);
      toast.error(errorMsg, { position: "top-center" });
      setLoading(false);
    } finally {
      setLoading(false);
    }
  };

  const formik = useFormik({
    initialValues: {
      first_name: "",
      last_name: "",
      email: "",
      contact_number: "",
      dob: format(new Date(), "MM-dd-yyyy"),
      timeslot_id: "",
      learning_style: "",
      address: "",
      city: "",
      state: "",
      zip_code: "",
      id_card_front: null,
      id_card_back: null,
    },
    validationSchema: Yup.object({
      first_name: Yup.string().required("First Name is required"),
      last_name: Yup.string().required("Last Name is required"),
      email: Yup.string().email("Invalid email").required("Email is required"),
      contact_number: Yup.string()
        .matches(/^\(\d{3}\)-\d{3}-\d{4}$/, "Contact Number is not valid")
        .required("Contact Number is required")
        .min(14, "Contact Number must be in the format (999)-999-9999"),
      dob: Yup.string().required("Date of birth is required"),
      timeslot_id: Yup.string().required("Please select a timeslot"),
      learning_style: Yup.string().required("Please select a learning style"),
      address: Yup.string().required("Please enter address"),
      city: Yup.string().required("Please enter city"),
      state: Yup.string().required("Please enter state"),
      zip_code: Yup.string().required("Please enter zip code"),
      id_card_front: Yup.mixed().required("Front ID file is required"),
      id_card_back: Yup.mixed().required("Back ID file is required"),
    }),
    onSubmit: (values) => {
      submitHandler(values);
    },
  });

  return (
    <div className="bg-[#000] w-full min-h-[100vh] registration-page px-3">
      <div className="relative z-10 pb-[30px] mb:pb-[35px]">
        <div className="py-6 max-w-[1200px] lg:px-4 mx-auto">
          <Link
            href="/"
            className="max-w-[120px] md:max-w-[180px] mx-auto block"
          >
            <Image
              src={Logo}
              alt="Agent Ambition Logo"
              className="max-w-[120px] md:max-w-[180px] mx-auto"
            />
          </Link>
        </div>

        {/* <h1 className="font-semibold text-[20px] md:text-[28px] text-center text-white mb-4">
          Just 24 students per class for a supportive experience
        </h1> */}
        <div className="flex justify-center items-center gap-2 sm:gap-3 md:gap-4 lg:gap-12 flex-wrap text-white mx-auto">
          <span className="text-base">
            <b className="text-[#D8A32B]">Schedule:</b> Monday-Thursday
          </span>
          <span className="text-base">
            <b className="text-[#D8A32B]">Duration:</b> 4 weeks course
          </span>
          <span className="text-base">
            <b className="text-[#D8A32B]">Location:</b> 2954 West 8 street
          </span>
        </div>
        <div className="max-w-7xl mx-auto relative z-2 text-white">
          <h1 className="font-semibold text-[20px] md:text-[28px] text-center my-4">
            What You Get When You Enroll
          </h1>
          <p>
            When you enroll, you’ll be guided by certified instructors who make learning simple and engaging. You’ll complete a state-approved 77-hour New York pre-licensing course designed to give you everything you need to succeed. Along the way, you’ll reinforce your knowledge through activities and practice questions that keep you sharp and confident. By the end, you’ll graduate with a course certificate and the preparation to ace the state exam and launch your real estate career.
          </p>
          
          {/* <h6 className="mb-1">Empowering Hook:</h6> */}
          <p className="font-bold">
            "This isn’t about memorizing laws and terms — it’s about building
            the confidence, skills, and mindset of a real estate professional.
            By the time you walk out, you won’t just be prepared to pass the
            exam — you’ll be ready to succeed."
          </p>
        </div>
        <div className="max-w-[1040px] registration-page-form-wrapper py-[20px] px-[20px] md:p-7 lg:p-10 mx-auto rounded-[32px] mt-[30px] md:mt-[35px]">
          <form onSubmit={formik.handleSubmit}>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-x-9 gap-y-4 mb-[30px]">
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  First Name:
                </label>
                <input
                  type="text"
                  name="first_name"
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  value={formik.values.first_name}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                  placeholder="Enter First Name"
                />
                {formik.touched.first_name && formik.errors.first_name && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.first_name}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Last Name:
                </label>
                <input
                  type="text"
                  name="last_name"
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  value={formik.values.last_name}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                  placeholder="Enter Last Name"
                />
                {formik.touched.last_name && formik.errors.last_name && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.last_name}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Email:
                </label>
                <input
                  type="email"
                  name="email"
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  value={formik.values.email}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                  placeholder="Enter Email"
                />
                {formik.touched.email && formik.errors.email && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.email}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Contact Number:
                </label>
                <IMaskInput
                  mask="(000)-000-0000"
                  value={formik.values.contact_number}
                  onAccept={(value) =>
                    formik.setFieldValue("contact_number", value)
                  } // Handle changes
                  onBlur={formik.handleBlur}
                  name="contact_number"
                  className="outline-none border rounded-lg px-3 py-3 bg-white border-gray-700 placeholder:text-xs"
                  placeholder="Enter Contact Number"
                />
                {formik.touched.contact_number &&
                  formik.errors.contact_number && (
                    <span className="text-red-500 text-sm">
                      {formik.errors.contact_number}
                    </span>
                  )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Date of Birth:
                </label>
                <DatePicker
                  selected={date}
                  onChange={(newDate) => {
                    if (newDate) {
                      formik.setFieldValue(
                        "dob",
                        format(newDate, "MM-dd-yyyy")
                      );
                      setDate(newDate);
                    }
                  }}
                  dateFormat="MM-dd-yyyy"
                  className="outline-none py-2 px-3 border border-grey rounded-md w-full h-[50px] text-[12px] cursor-pointer"
                  placeholderText="Select Date of Birth"
                  renderCustomHeader={({ date, changeYear, changeMonth }) => {
                    const currentYear = new Date().getFullYear();
                    return (
                      <div className="flex justify-center gap-2 p-2">
                        {/* Year dropdown */}
                        <select
                          className="border rounded px-2 py-1 text-sm"
                          value={date.getFullYear()}
                          onChange={({ target: { value } }) =>
                            changeYear(value)
                          }
                        >
                          {Array.from({ length: 200 }, (_, i) => (
                            <option key={i} value={currentYear - i}>
                              {currentYear - i}
                            </option>
                          ))}
                        </select>

                        {/* Month dropdown */}
                        <select
                          className="border rounded px-2 py-1 text-sm"
                          value={date.getMonth()}
                          onChange={({ target: { value } }) =>
                            changeMonth(value)
                          }
                        >
                          {[
                            "Jan",
                            "Feb",
                            "Mar",
                            "Apr",
                            "May",
                            "Jun",
                            "Jul",
                            "Aug",
                            "Sep",
                            "Oct",
                            "Nov",
                            "Dec",
                          ].map((month, i) => (
                            <option key={i} value={i}>
                              {month}
                            </option>
                          ))}
                        </select>
                      </div>
                    );
                  }}
                />

                {formik.touched.dob && formik.errors.dob && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.dob}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Preferred Timeslot:
                </label>
                <select
                  name="timeslot_id"
                  value={formik.values.timeslot_id}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                >
                  <option value="">--select preferred timeslot--</option>
                  <option value="1">10:00 AM to 4:00 PM</option>
                  <option value="2">5:00 PM to 9:30 PM</option>
                  {/* <option value="1">11:30 AM to 1:00 PM</option>
                  <option value="2">2:30 PM to 4:00 PM</option>
                  <option value="3">5:30 PM to 7:00 PM</option>
                  <option value="3">7:30 PM to 9:00 PM</option> */}
                </select>
                {formik.touched.timeslot_id && formik.errors.timeslot_id && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.timeslot_id}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Choose Your Learning Style:
                </label>
                <select
                  name="learning_style"
                  value={formik.values.learning_style}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                >
                  <option value="">--select learning style--</option>
                  <option value="in_person">In-Person Classroom</option>
                  <option value="online">Live Online (Zoom)</option>
                </select>
                {formik.touched.learning_style &&
                  formik.errors.learning_style && (
                    <span className="text-red-500 text-sm">
                      {formik.errors.learning_style}
                    </span>
                  )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Address:
                </label>
                <input
                  type="text"
                  name="address"
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  value={formik.values.address}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                  placeholder="Enter Address"
                />
                {formik.touched.address && formik.errors.address && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.address}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  City:
                </label>
                <input
                  type="text"
                  name="city"
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  value={formik.values.city}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                  placeholder="Enter City"
                />
                {formik.touched.city && formik.errors.city && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.city}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  State:
                </label>
                <input
                  type="text"
                  name="state"
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  value={formik.values.state}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                  placeholder="Enter state"
                />
                {formik.touched.state && formik.errors.state && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.state}
                  </span>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Zip Code:
                </label>
                <input
                  type="text"
                  name="zip_code"
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  value={formik.values.zip_code}
                  className="outline-none py-2 px-3 border border-grey rounded-md h-[45px] md:h-[50px] text-[12px]"
                  placeholder="Enter Zip Code"
                />
                {formik.touched.zip_code && formik.errors.zip_code && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.zip_code}
                  </span>
                )}
              </div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-x-9 gap-y-4">
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Upload Front of ID:{" "}
                  <span className="opacity-50 text-sm">(JPG, PNG, or PDF)</span>
                </label>
                <div
                  className="border-2 border-dashed border-grey rounded-md h-[150px] flex flex-col items-center justify-center text-white text-sm cursor-pointer bg-[#1a1a1a] hover:bg-[#2a2a2a] transition-colors p-4"
                  onDragOver={handleDragOver}
                  onDrop={(e) => handleDrop(e, "front")}
                  onClick={() => frontFileInputRef.current.click()}
                >
                  <p>
                    Drag & drop a JPG, PNG, or PDF file here or click to upload
                  </p>
                  <input
                    type="file"
                    name="id_card_front"
                    accept="image/jpeg,image/png,application/pdf"
                    onChange={(e) =>
                      handleFileChange(e.target.files[0], "front")
                    }
                    ref={frontFileInputRef}
                    className="hidden"
                  />
                </div>
                {formik.touched.id_card_front &&
                  formik.errors.id_card_front && (
                    <span className="text-red-500 text-sm">
                      {formik.errors.id_card_front}
                    </span>
                  )}
                {frontImagePreview && (
                  <div className="mt-2 flex items-center gap-3">
                    <Image
                      src={frontImagePreview}
                      alt="Front ID Preview"
                      width={200}
                      height={150}
                      className="object-cover rounded-md"
                    />
                    <button
                      type="button"
                      onClick={() => removeFile("front")}
                      className="text-red-500 font-bold text-sm hover:underline"
                    >
                      Delete
                    </button>
                  </div>
                )}
                {frontFileName && !frontImagePreview && (
                  <div className="mt-2 flex items-center gap-3 text-white text-sm">
                    <span>Uploaded file: {frontFileName}</span>
                    <button
                      type="button"
                      onClick={() => removeFile("front")}
                      className="text-red-500 font-bold text-sm hover:underline"
                    >
                      Delete
                    </button>
                  </div>
                )}
              </div>
              <div className="flex flex-col">
                <label className="text-white text-[14px] md:text-[16px] font-bold mb-[10px]">
                  Upload Back of ID:{" "}
                  <span className="opacity-50 text-sm">(JPG, PNG, or PDF)</span>
                </label>
                <div
                  className="border-2 border-dashed border-grey rounded-md h-[150px] flex flex-col items-center justify-center text-white text-sm cursor-pointer bg-[#1a1a1a] hover:bg-[#2a2a2a] transition-colors p-4"
                  onDragOver={handleDragOver}
                  onDrop={(e) => handleDrop(e, "back")}
                  onClick={() => backFileInputRef.current.click()}
                >
                  <p>
                    Drag & drop a JPG, PNG, or PDF file here or click to upload
                  </p>
                  <input
                    type="file"
                    name="id_card_back"
                    accept="image/jpeg,image/png,application/pdf"
                    onChange={(e) =>
                      handleFileChange(e.target.files[0], "back")
                    }
                    ref={backFileInputRef}
                    className="hidden"
                  />
                </div>
                {formik.touched.id_card_back && formik.errors.id_card_back && (
                  <span className="text-red-500 text-sm">
                    {formik.errors.id_card_back}
                  </span>
                )}
                {backImagePreview && (
                  <div className="mt-2 flex items-center gap-3">
                    <Image
                      src={backImagePreview}
                      alt="Back ID Preview"
                      width={200}
                      height={150}
                      className="object-cover rounded-md"
                    />
                    <button
                      type="button"
                      onClick={() => removeFile("back")}
                      className="text-red-500 font-bold text-sm hover:underline"
                    >
                      Delete
                    </button>
                  </div>
                )}
                {backFileName && !backImagePreview && (
                  <div className="mt-2 flex items-center gap-3 text-white text-sm">
                    <span>Uploaded file: {backFileName}</span>
                    <button
                      type="button"
                      onClick={() => removeFile("back")}
                      className="text-red-500 font-bold text-sm hover:underline"
                    >
                      Delete
                    </button>
                  </div>
                )}
              </div>
            </div>
            <div className="mb-5 mt-10">
              {errorMessage && (
                <span className="text-red-500 font-bold block mb-[12px]">
                  {errorMessage}
                </span>
              )}
              <button
                type="submit"
                disabled={loading}
                className={`bg-gradient-to-t from-[#B9802A] to-[#D8A32B] text-white py-2 px-3 rounded-md w-full h-[50px] flex items-center justify-center ${
                  loading ? "opacity-70 cursor-not-allowed" : ""
                }`}
              >
                {loading ? "Submitting..." : "Register Now"}
              </button>
            </div>
            {/* <div className="text-center text-white text-sm font-semibold">
              *First 24 students will receive a promo for 24% Discount
            </div> */}
          </form>
        </div>
      </div>
    </div>
  );
};

export default RegistretionForm;
